package furny.swing.admin.importer;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.filechooser.FileFilter;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;

import org.jdesktop.swingx.JXHyperlink;

import com.jme3.app.SimpleApplication;
import com.jme3.app.state.AppState;

import furny.furndb.importer.FurnDBImporter;
import furny.furndb.importer.FurnDBImporter.ExitMode;
import furny.furndb.importer.FurnDBImporter.ImportListener;
import furny.swing.admin.viewer.ModelViewerState;
import furny.swing.common.FileChooserPanel;
import furny.swing.common.FileChosenListener;
import furny.util.FileUtils;

/**
 * Panel for importing 3d models from OgreXML.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class ImporterPanel extends JPanel {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(ImporterPanel.class
      .getName());

  private FileChooserPanel fileChooserPanel;
  private FileListModel model;
  private JList filesList;
  private final File assetBaseDir = new File("src/assets/");
  private File watchDir = new File("src/assets/models");

  private FileWatchThread watchThread;
  private boolean watchThreadActive;
  private ActionRefreshNow actionRefreshNow;

  private final FurnDBImporter importer;
  private final SimpleApplication app;

  /**
   * Instantiates a new importer panel.
   * 
   * @param app
   *          the simple application
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ImporterPanel(final SimpleApplication app) {
    this.app = app;
    importer = new FurnDBImporter(app);
    importer.setExitMode(ExitMode.DETACH_AND_UPDATE_IDS);

    setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.gridwidth = 1;

    add(createLiveExchangePanel(), constraints);
  }

  /**
   * Creates the live exchange panel.
   * 
   * @return the j panel
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private JPanel createLiveExchangePanel() {
    final JPanel panel = new JPanel();
    panel.setBorder(BorderFactory.createTitledBorder("Directory import"));

    panel.setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 0.0d;
    constraints.weighty = 0.0d;
    constraints.gridwidth = 1;
    constraints.fill = GridBagConstraints.NONE;

    final JCheckBox cb = new JCheckBox(new ActionToggleWatchDog());
    cb.setSelected(watchThreadActive);
    panel.add(cb, constraints);

    constraints.gridx++;

    actionRefreshNow = new ActionRefreshNow();
    actionRefreshNow.setEnabled(!watchThreadActive);
    panel.add(new JButton(actionRefreshNow), constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.gridwidth = 3;

    constraints.fill = GridBagConstraints.HORIZONTAL;

    fileChooserPanel = new FileChooserPanel(new WatchDirChosenListener(),
        new DirectoryFileFilter());
    fileChooserPanel.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

    panel.add(fileChooserPanel, constraints);

    constraints.gridy++;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.fill = GridBagConstraints.BOTH;

    model = new FileListModel();
    filesList = new JList(model);
    panel.add(new JScrollPane(filesList), constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.weightx = 0.0d;
    constraints.weighty = 0.0d;
    constraints.fill = GridBagConstraints.NONE;
    constraints.gridwidth = 1;

    panel.add(new JButton(new ActionImport()), constraints);

    // added this because there is a bug in the jvm that causes a crash
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        fileChooserPanel.setFile(watchDir);
        if (watchThreadActive) {
          watchThread = new FileWatchThread();
          watchThread.start();
        }
      }
    });

    return panel;
  }

  /**
   * Creates the import panel.
   * 
   * @param model
   *          the file table model
   * @return the j panel
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private JPanel createImportPanel(final FileTableModel model) {
    final JPanel panel = new JPanel();

    panel.setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 1.0d;
    constraints.weighty = 0.0d;
    constraints.gridwidth = 1;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.EAST;

    panel.add(Box.createHorizontalGlue(), constraints);

    constraints.weightx = 0.0d;
    constraints.gridx++;

    panel.add(new JXHyperlink(new ActionSelectNone(model)), constraints);

    constraints.gridx++;

    panel.add(new JXHyperlink(new ActionSelectAll(model)), constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.gridwidth = 3;
    constraints.fill = GridBagConstraints.BOTH;

    final JTable table = new JTable(model);
    table.setDefaultRenderer(File.class, new FileTableRenderer());
    panel.add(new JScrollPane(table), constraints);

    return panel;
  }

  /**
   * File filter that displays only directories.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class DirectoryFileFilter extends FileFilter {
    @Override
    public String getDescription() {
      return "Directories";
    }

    @Override
    public boolean accept(final File f) {
      return true;
    }
  }

  /**
   * Listener that will be notified about a directory being chosen.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class WatchDirChosenListener implements FileChosenListener {
    @Override
    public void fileChosen(final File file, final int type) {
      watchDir = file;
      model.update();
    }
  }

  /**
   * Action to deselect all table entries.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class ActionSelectNone extends AbstractAction {
    private final FileTableModel model;

    /**
     * Instantiates a new action to deselect all.
     * 
     * @param model
     *          the model
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionSelectNone(final FileTableModel model) {
      super("None");
      this.model = model;
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      for (final FileEntry entry : model.getEntries()) {
        entry.setSelected(false);
      }
      model.fireTableDataChanged();
    }
  }

  /**
   * Action to select all table entries.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class ActionSelectAll extends AbstractAction {
    private final FileTableModel model;

    /**
     * Instantiates a new action to select all.
     * 
     * @param model
     *          the model
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionSelectAll(final FileTableModel model) {
      super("All");
      this.model = model;
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      for (final FileEntry entry : model.getEntries()) {
        entry.setSelected(true);
      }
      model.fireTableDataChanged();
    }
  }

  /**
   * Action to activate/deactivate the directory watchdog.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleWatchDog extends AbstractAction {

    /**
     * Instantiates a new action to toggle the watch dog.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleWatchDog() {
      super("Auto refresh");
      putValue(SHORT_DESCRIPTION,
          "Refresh the directory view when something changes");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      watchThreadActive = ((AbstractButton) e.getSource()).isSelected();

      actionRefreshNow.setEnabled(!watchThreadActive);

      if (watchThreadActive) {
        watchThread = new FileWatchThread();
        watchThread.start();
      }
    }
  }

  /**
   * Action to refresh the current directory.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionRefreshNow extends AbstractAction {

    /**
     * Instantiates a new action refresh now.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionRefreshNow() {
      super("Refresh now");
      putValue(SHORT_DESCRIPTION, "Refresh the directory view now");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      model.update();
    }
  }

  /**
   * Action to start the file import.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionImport extends AbstractAction {

    /**
     * Instantiates a new action to start the import.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionImport() {
      super("Import...");
      putValue(SHORT_DESCRIPTION, "Import all furniture models from directory");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      LOGGER.log(Level.INFO, "absolute: " + watchDir.getAbsolutePath());

      String assetPath = watchDir.getAbsolutePath()
      // .replace(assetBaseDir.getAbsolutePath(), "")
          .replace('\\', '/');

      LOGGER.log(Level.INFO, "1: " + assetPath);

      if (assetPath.startsWith("/") && assetPath.length() > 1) {
        assetPath = assetPath.substring(1);
      }

      LOGGER.log(Level.INFO, "2: " + assetPath);

      if (!assetPath.endsWith("/") && assetPath.length() > 0) {
        assetPath = assetPath + "/";
      }

      LOGGER.log(Level.INFO, "3: " + assetPath);

      final String fAssetPath = assetPath;

      final FileTableModel model = new FileTableModel(assetPath);

      if (model.getRowCount() == 0) {
        JOptionPane.showMessageDialog(ImporterPanel.this,
            "No model files found in this directory.", "No files to import",
            JOptionPane.INFORMATION_MESSAGE);
        return;
      }

      final int n = JOptionPane.showConfirmDialog(ImporterPanel.this,
          createImportPanel(model), "Select files to import",
          JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);

      final List<File> files = model.getSelectedFiles();

      if (n == JOptionPane.OK_OPTION && !files.isEmpty()) {

        final SwingWorker<Void, Void> worker = new SwingWorker<Void, Void>() {
          @Override
          protected Void doInBackground() throws Exception {

            // if there is a viewer state, disable it.
            final AppState viewerState = app.getStateManager().getState(
                ModelViewerState.class);
            if (viewerState != null) {
              viewerState.setEnabled(false);
            }

            importer.importModelsFromFiles(fAssetPath, files,
                new ImportListener() {

                  @Override
                  public void importDone() {

                    // if there is a viewer state, enable it.
                    final AppState viewerState = app.getStateManager()
                        .getState(ModelViewerState.class);

                    LOGGER.info("viewer activated" + viewerState);

                    if (viewerState != null) {

                      viewerState.setEnabled(true);
                    }

                    SwingUtilities.invokeLater(new Runnable() {
                      @Override
                      public void run() {
                        JOptionPane.showMessageDialog(ImporterPanel.this,
                            "Import was successful.", "Import successful",
                            JOptionPane.INFORMATION_MESSAGE);
                      }
                    });
                  }
                });

            return null;
          }

        };

        worker.execute();
      }
    }
  }

  /**
   * List model that displays the content of a directory.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class FileListModel extends DefaultListModel {
    private final List<File> subFiles = new ArrayList<File>();

    /**
     * Instantiates a new file list model.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public FileListModel() {
      update();
    }

    /**
     * Update the directory view.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    private void update() {
      final List<File> actFiles = FileUtils.getFiles(watchDir,
          ".*\\.mesh\\.xml");
      subFiles.clear();
      subFiles.addAll(actFiles);

      fireContentsChanged(this, 0, subFiles.size() - 1);
    }

    @Override
    public Object get(final int index) {
      return subFiles.get(index).getName();
    }

    @Override
    public int getSize() {
      return subFiles.size();
    }

    @Override
    public Object getElementAt(final int index) {
      return get(index);
    }
  }

  /**
   * Table model that displays the content of a directory.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class FileTableModel extends AbstractTableModel {
    private static final int COL_FILE = 0;
    private static final int COL_SELECT = 1;
    private static final int COL_COUNT = 2;

    private final List<FileEntry> entries = new ArrayList<FileEntry>();

    private final String assetPath;

    /**
     * Instantiates a new file table model.
     * 
     * @param assetPath
     *          the asset path
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public FileTableModel(final String assetPath) {
      this.assetPath = assetPath;
      update();

    }

    /**
     * Update the directory view.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    private void update() {
      entries.clear();
      final List<File> files = FileUtils.getFiles(new File(assetPath),
          ".*\\.mesh\\.xml");

      for (final File file : files) {
        entries.add(new FileEntry(file));
      }
    }

    /**
     * Gets the file entries.
     * 
     * @return the entries
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public List<FileEntry> getEntries() {
      return entries;
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex) {
      switch (columnIndex) {
      case COL_FILE:
        return File.class;
      case COL_SELECT:
        return Boolean.class;
      default:
        return String.class;
      }
    }

    @Override
    public boolean isCellEditable(final int row, final int column) {
      return column == COL_SELECT;
    }

    @Override
    public void setValueAt(final Object aValue, final int row, final int column) {
      if (column == COL_SELECT) {
        entries.get(row).setSelected((Boolean) aValue);
      }
    }

    @Override
    public int getColumnCount() {
      return COL_COUNT;
    }

    @Override
    public int getRowCount() {
      return entries.size();
    }

    @Override
    public Object getValueAt(final int row, final int column) {
      switch (column) {
      case COL_FILE:
        return entries.get(row).getFile();
      case COL_SELECT:
        return entries.get(row).isSelected();
      default:
        return "";
      }
    }

    @Override
    public String getColumnName(final int column) {
      switch (column) {
      case COL_FILE:
        return "File";
      case COL_SELECT:
        return "Import";
      default:
        return "";
      }
    }

    /**
     * Gets the selected files.
     * 
     * @return the selected files
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public List<File> getSelectedFiles() {
      final List<File> files = new ArrayList<File>();

      for (final FileEntry entry : entries) {
        if (entry.isSelected()) {
          files.add(entry.getFile());
        }
      }

      return files;
    }
  }

  /**
   * Renderer for the file table.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class FileTableRenderer extends DefaultTableCellRenderer {
    @Override
    public Component getTableCellRendererComponent(final JTable table,
        final Object value, final boolean isSelected, final boolean hasFocus,
        final int row, final int column) {

      Object val;
      if (value instanceof File) {
        val = ((File) value).getName();
      } else {
        val = value;
      }

      return super.getTableCellRendererComponent(table, val, isSelected,
          hasFocus, row, column);
    }
  }

  /**
   * An entry that encapsulates information about a file.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class FileEntry {
    private final File file;
    private boolean selected;

    /**
     * Instantiates a new file entry.
     * 
     * @param file
     *          the file
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public FileEntry(final File file) {
      super();
      this.file = file;
    }

    /**
     * Sets the entry selected.
     * 
     * @param selected
     *          the new selected
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void setSelected(final boolean selected) {
      this.selected = selected;
    }

    /**
     * Checks if the entry is selected.
     * 
     * @return <code>true</code>, if is selected
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public boolean isSelected() {
      return selected;
    }

    /**
     * Gets the file.
     * 
     * @return the file
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public File getFile() {
      return file;
    }
  }

  /**
   * Watchdog that waits for new files in the directory.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class FileWatchThread extends Thread {

    /**
     * Instantiates a new file watch thread.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public FileWatchThread() {
      setDaemon(true);
      setPriority(NORM_PRIORITY - 2);
    }

    @Override
    public void run() {
      while (watchThreadActive) {
        try {
          sleep(2000);
        } catch (final InterruptedException e) {
          LOGGER.log(Level.SEVERE, "Failed to sleep", e);
        }

        if (filesList.isShowing()) {
          SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
              model.update();
            }
          });
        }
      }
    }
  }
}
